  // Problem description: //<>//
  // Deformable object simulation
import peasy.*;

// Display control:

PeasyCam _camera;   // Mouse-driven 3D camera

// Simulation and time control:

float _timeStep;              // Simulation time-step (s)
int _lastTimeDraw = 0;        // Last measure of time in draw() function (ms)
float _deltaTimeDraw = 0.0;   // Time between draw() calls (s)
float _simTime = 0.0;         // Simulated time (s)
float _elapsedTime = 0.0;     // Elapsed (real) time (s)

// Output control:

boolean _writeToFile = false;
PrintWriter _output;

// System variables:

Ball _ball;                           // Sphere
// DeformableObject _defOb;              // Deformable object
SpringLayout _springLayout;           // Current spring layout
float ball_vel = 500;
boolean _displaySprings = false;
boolean _paused = false;
float _Ke;
float _Kd;

// Banderas
ArrayList<DeformableObject> banderas;
ArrayList<Post> postes;
boolean activeGravity = false;

boolean windEnabled = true;
PVector Fwind1 = new PVector( 1, 5, 0.5);
float fwind1 = 5;
PVector Fwind;
float fwind = 5;

boolean ctrl = false;
boolean shift = false;
  
  // Main code:

  void settings()
  {
    size(DISPLAY_SIZE_X, DISPLAY_SIZE_Y, P3D);
  }

  void setup()
  {
    frameRate(DRAW_FREQ);
    _lastTimeDraw = millis();

    float aspect = float(DISPLAY_SIZE_X)/float(DISPLAY_SIZE_Y);
    perspective((FOV*PI)/180, aspect, NEAR, FAR);
    _camera = new PeasyCam(this, 0);
    _camera.rotateX(PI*5/12);
    _camera.rotateZ(PI);
    _camera.setDistance(1000);

    _springLayout = SpringLayout.STRUCTURAL_AND_SHEAR_AND_BEND;

    initSimulation();
  }

  void stop()
  {
    endSimulation();
  }

  void initSimulation()
  {
    println( "initSimulation()");
    if (_writeToFile)
    {
      _output = createWriter(FILE_NAME);
      writeToFile("t, n, Tsim");
    }
    _Ke = Ke;
    _Kd = Kd;

    _simTime = 0.0;
    _timeStep = TS*TIME_ACCEL;
    _elapsedTime = 0.0;

    banderas = new ArrayList<DeformableObject>();
    postes = new ArrayList<Post>();
    // goalPosts = new ArrayList<Post>();

    Fwind = new PVector();

    float altura = 200+sepV*2*nodesV/2.0;
    buildBandera( new PVector( sepH*nodesH, 0, altura), SpringLayout.STRUCTURAL, #ff0000);
    buildBandera( new PVector( 0, 0, altura), SpringLayout.STRUCTURAL_AND_BEND, #00ff00);
    buildBandera( new PVector( -sepH*nodesH, 0, altura), SpringLayout.STRUCTURAL_AND_SHEAR, #0000ff);

    Post post = new Post( new PVector( 0, 0, altura+(nodesV/2.0*sepV)), 1000, 5, 5);
    postes.add( post);

    PVector s = new PVector();
    s.set( 0,-500, B_R*10);
    _ball = new Ball( s, new PVector() , B_M, B_R, activeGravity, #ff0000, true, false);
  }

  void buildBandera( PVector s, SpringLayout layout, color c){
    boolean [][] willBeClamped;

    willBeClamped = new boolean[nodesH][nodesV];
    for( int i = 0; i < nodesH; i++){
      for( int j = 0; j < nodesV; j++){
        willBeClamped[i][j] = false;
      }
    }

    willBeClamped[0][nodesV-1] = true;
    willBeClamped[nodesH-1][nodesV-1] = true;

    PVector H = new PVector( -1, 0, 0);
    PVector V = new PVector( 0, 0, 1);

    DeformableObject _defOb = new DeformableObject( s, nodesH, nodesV, sepH, sepV, H, V, willBeClamped, layout, c);
    banderas.add( _defOb);
  }

  void resetBall() // Deja Bola en Posicion y velocidad inicial
  {
    PVector pos = new PVector( 0, 0, 0);
    _ball.setPos( pos);
  }

  void restartBall() // Deja Bola en Posicion inicial
  {
    PVector pos = new PVector( 0,-500, B_R*10);
    _ball.setPos( pos);
  }

  void restartSimulation()
  {
    _simTime = 0.0;
    _timeStep = TS*TIME_ACCEL;
    _elapsedTime = 0.0;
  }

  void endSimulation()
  {
    if (_writeToFile)
    {
      _output.flush();
      _output.close();
    }
  }

  int getTotalNodes(){
    int nodes = 0;
    for( DeformableObject df : banderas){
      nodes += df.getNumNodes();
    }
    return nodes;
  }

  int getTotalBrokenSprings(){
    int broken = 0;
    for( DeformableObject df : banderas){
      broken += df.getNumBrokenSprings();
    }
    return broken;
  }

  void updateWind(){
    PVector newWind = new PVector(
      (Fwind1.x != 0) ? 3 + random(0, Fwind1.x) : 0,
      (Fwind1.y != 0) ? 1 + random(0, Fwind1.y)*2 : 0,
      Fwind1.z
    );
    Fwind.set( newWind.normalize());
  }

  void draw()
  {
    int now = millis();
    _deltaTimeDraw = (now - _lastTimeDraw)/1000.0;
    _elapsedTime += _deltaTimeDraw;
    _lastTimeDraw = now;

    background(BACKGROUND_COLOR);
    drawStaticEnvironment();
    drawDynamicEnvironment();

    if (REAL_TIME)
    {
      float expectedSimulatedTime = TIME_ACCEL*_deltaTimeDraw;
      float expectedIterations = expectedSimulatedTime/_timeStep;
      int iterations = 0;

      for (; iterations < floor(expectedIterations); iterations++)
        updateSimulation();

      if ((expectedIterations - iterations) > random(0.0, 1.0))
      {
        updateSimulation();
        iterations++;
      }
    } 
    else
      updateSimulation();

    displayInfo();

    if (_writeToFile)
      writeToFile(_simTime + "," + getTotalNodes() + "," +_deltaTimeDraw);
  }

  void drawStaticEnvironment()
  {
    noStroke();
    fill(255, 0, 0);
    box(1000.0, 1.0, 1.0);

    fill(0, 255, 0);
    box(1.0, 1000.0, 1.0);

    fill(0, 0, 255);
    box(1.0, 1.0, 1000.0);

    fill(255, 255, 255);
    sphere(1.0);

    pushMatrix();{
      fill( #238229);
      box( 1000, 1000, 0);
    }
    popMatrix();

    for( Post p : postes){
      p.display();
    }
  }

  void drawDynamicEnvironment()
  {
    // _defOb.render();
    _ball.render();
    for( DeformableObject df : banderas)
      df.render();
  }

  void updateSimulation()
  {
    if( _paused) return;

    if( windEnabled)
      updateWind();

    for( DeformableObject df : banderas){
      df.update( _timeStep);
    }

    _ball.update( _timeStep);

    computeCollisions();

    _simTime += _timeStep;
  }

  void computeCollisions(){
    for( DeformableObject df : banderas){
      df.computeCollisions();
    }
    _ball.computeColisionSpringFloor();
  }

  void writeToFile(String data)
  {
    _output.println(data);
  }

  void keyReleased(){
    if( keyCode == 17)
      ctrl = false;
    if( keyCode == 16)
      shift = false;
  }

  void keyPressed()
  {
    if (key == '1'){
      // restartSimulation(SpringLayout.STRUCTURAL);
      _springLayout = SpringLayout.STRUCTURAL;
      // _defOb.updateSpringLayout();
      for( DeformableObject df : banderas)
        df.updateSpringLayout( SpringLayout.STRUCTURAL);
    }

    if (key == '2'){
      // restartSimulation(SpringLayout.SHEAR);
      _springLayout = SpringLayout.SHEAR;
      // _defOb.updateSpringLayout();
      for( DeformableObject df : banderas)
        df.updateSpringLayout( SpringLayout.SHEAR);
    }

    if (key == '3'){
      // restartSimulation(SpringLayout.BEND);
      _springLayout = SpringLayout.BEND;
      // _defOb.updateSpringLayout();
      for( DeformableObject df : banderas)
        df.updateSpringLayout( SpringLayout.BEND);
    }

    if (key == '4'){
      // restartSimulation(SpringLayout.STRUCTURAL_AND_SHEAR);
      _springLayout = SpringLayout.STRUCTURAL_AND_SHEAR;
      // _defOb.updateSpringLayout();
      for( DeformableObject df : banderas)
        df.updateSpringLayout( SpringLayout.STRUCTURAL_AND_SHEAR);
    }

    if (key == '5'){
      // restartSimulation(SpringLayout.STRUCTURAL_AND_BEND);
      _springLayout = SpringLayout.STRUCTURAL_AND_BEND;
      // _defOb.updateSpringLayout();
      for( DeformableObject df : banderas)
        df.updateSpringLayout( SpringLayout.STRUCTURAL_AND_BEND);
    }

    if (key == '6'){
      // restartSimulation(SpringLayout.SHEAR_AND_BEND);
      _springLayout = SpringLayout.SHEAR_AND_BEND;
      // _defOb.updateSpringLayout();
      for( DeformableObject df : banderas)
        df.updateSpringLayout( SpringLayout.SHEAR_AND_BEND);
    }

    if (key == '7'){
      // restartSimulation(SpringLayout.STRUCTURAL_AND_SHEAR_AND_BEND);
      _springLayout = SpringLayout.STRUCTURAL_AND_SHEAR_AND_BEND;
      // _defOb.updateSpringLayout();
      for( DeformableObject df : banderas)
        df.updateSpringLayout( SpringLayout.STRUCTURAL_AND_SHEAR_AND_BEND);
    }

    if (key == 'r' || key == 'R')
      initSimulation();

    if(key == 'b' || key == 'B')
      restartBall();

    if( key == 's' || key == 'S')
      _displaySprings = !_displaySprings;

    if( key == 'g' || key == 'G'){
      activeGravity = !activeGravity;
      _ball.setGravity( activeGravity);
    }

    if( keyCode == 17)
      ctrl = true;
    if( keyCode == 16)
      shift = true;

    if (keyCode == UP){
      if( ctrl)
        _ball.addVel( new PVector( 0, 0, 1f).mult( ball_vel));
      else if( shift)
        _ball.addPos( new PVector( 0, 1.5f, 0));
      else
        _ball.addVel( new PVector( 0, 1f, 0).mult( ball_vel));
    }

    if (keyCode == DOWN){
      if( ctrl)
        _ball.addVel( new PVector( 0, 0, -1f).mult( ball_vel));
      else if( shift)
        _ball.addPos( new PVector( 0, -1.5f, 0));
      else
        _ball.addVel( new PVector( 0, -1f, 0).mult( ball_vel));
      
    }

    if (keyCode == LEFT){
      if( shift)
        _ball.addPos( new PVector( 1.5f, 0, 0));
      else
      _ball.addVel( new PVector( 1f, 0, 0).mult( ball_vel));
    }

    if (keyCode == RIGHT){
      if( shift)
        _ball.addPos( new PVector( -1.5f, 0, 0));
      else
        _ball.addVel( new PVector( -1f, 0, 0).mult( ball_vel));
    }

    if( key == 'w' || key == 'W')
      windEnabled = !windEnabled;

    if( key == 'c' || key == 'C')
      _ball.toggleClamped();

    if( key == '+')
      ball_vel *= 1.5;

    if( key == '-')
      ball_vel /= 1.5;

    if (key == 'D' || key == 'd')
      DRAW_MODE = !DRAW_MODE;

    if (key == 'I' || key == 'i')
      initSimulation();

    if (key == 'n' || key == 'N')
      fwind/=1.05;

    if (key == 'm' || key == 'M')
      fwind*=1.05;

    if (key == ' ')
      _paused = !_paused;

    if (key == '.')
      _Ke *= 1.15;

    if (key == ',')
      _Ke /= 1.15;
  }

  void buildPosts(){
    
  }

  void displayInfo()
  {
    float ch = 0.02;
    int i = 1;
    pushMatrix();
    {
      camera();
      fill(0);
      textSize(20);
      textAlign( LEFT);
      text("Frame rate = " + nf( 1.0/_deltaTimeDraw, 1, 2) + " fps", width*0.025, i*height*ch);
      i++;
      text("Elapsed time = " + nf( _elapsedTime, 1, 2) + " s", width*0.025, i*height*ch);
      i++;
      text("Simulated time = " + nf( _simTime, 1, 2) + " s ", width*0.025, i*height*ch);
      i++;
      text("Spring layout = " + _springLayout, width*0.025, i*height*ch);
      i++;
      text("Ball current velocity = " + _ball.getVel().mult( 0.01) + " m/s", width*0.025, i*height*ch);
      i++;
      text("Ball mass: " + _ball._m + " Kg", width*0.025, i*height*ch);
      i++;
      text("Particle mass: " + P_M + " Kg", width*0.025, i*height*ch);
      i++;
      text("Ball clamped: " + ( _ball._clamped ? "true" : "false"), width*0.025, i*height*ch);
      i++;
      text("Ball gravity: " + ( _ball._gravity ? "true" : "false"), width*0.025, i*height*ch);
      i++;
      text("Ke: " + _Ke, width*0.025, i*height*ch);
      i++;
      text("Kdd: " + Kd, width*0.025, i*height*ch); 
      i++;
      text("Broken Springs: " + getTotalBrokenSprings(), width*0.025, i*height*ch);
      i++;
      text("Max Force: " + MAX_FORCE, width*0.025, i*height*ch);
      i++;
      PVector viento = PVector.mult( Fwind, fwind);
      text("Wind: " + (windEnabled ? "[" + nf(viento.x, 0, 2) + ", " + nf( viento.y, 0, 2) + ", " + nf(viento.z, 0, 2) + "] Mag: " + nf( viento.mag(), 0, 2) : "Disabled"), width*0.025, i*height*ch);

      //Controls
      i = 1;
      textAlign( RIGHT);
      text("CONTROLS: ", width-width*0.025, i*height*ch);
      i++;
      text("[SPACE] " + (_paused ? "Resume": "Pause"), width-width*0.025, i*height*ch);
      i++;
      text("[UP / DOWN / LEFT / RIGHT] Move Ball", width-width*0.025, i*height*ch);
      i++;
      text("[CTRL + UP/DOWN] Move Ball Vertically", width-width*0.025, i*height*ch);
      i++;
      text("[SHIFT] Move Slow", width-width*0.025, i*height*ch);
      i++;
      text("[+ / -] Increase / Decrease Ball velocity", width-width*0.025, i*height*ch);
      i++;
      text("[. / ,] Increase / Decrease Ke", width-width*0.025, i*height*ch);
      i++;
      text("[B] Restart Ball Position", width-width*0.025, i*height*ch);
      i++;
      text("[G] Toggle Ball Gravity", width-width*0.025, i*height*ch);
      i++;
      text("[C] Toggle Ball Clamped", width-width*0.025, i*height*ch);
      i++;
      text("[I] Init Simulation", width-width*0.025, i*height*ch);
      i++;
      text("[D] Change DRAW_MODE", width-width*0.025, i*height*ch);
      i++;
      text("[S] Toggle Springs Display", width-width*0.025, i*height*ch);
      i++;
      text("[N / M] Decrease / Increase Wind", width-width*0.025, i*height*ch);
      i++;
      text("[W] Toggle Wind", width-width*0.025, i*height*ch);

      i = 1;
      textAlign( CENTER);
      text("Red: STRUCTURAL", width/2, height-(i*height*ch));
      i++;
      text("Green: STRUCTURAL+BEND", width/2, height-(i*height*ch));
      i++;
      text("Blue: STRUCTURAL+SHEAR", width/2, height-(i*height*ch));
      
    }
    popMatrix();
  }
