public class DeformableObject
{
  int _numNodesH;   // Number of nodes in horizontal direction
  int _numNodesV;   // Number of nodes in vertical direction

  float _sepH;      // Separation of the object's nodes in the X direction (m)
  float _sepV;      // Separation of the object's nodes in the Y direction (m)

  // SpringLayout _springLayout;   // Physical layout of the springs that define the surface of each layer
  color _color;                 // Color (RGB)

  Particle[][] _nodes;                             // Particles defining the object
  PVector[][] _normals;
  ArrayList<DampedSpring> _springs;                // Springs joining the particles

  DeformableObject( PVector iniPos, int numNodesH, int numNodesV, float sepH, float sepV, PVector dirH, PVector dirV, boolean[][] willBeClamped, SpringLayout layout, color c){
    /*
     * Pasar en lugar de el punto Origen, 4 puntos referentes a 
     */
    this._numNodesH = numNodesH;
    this._numNodesV = numNodesV;
    this._sepH = sepH;
    this._sepV = sepV;
    this._color = c;

    _nodes = new Particle[this._numNodesH][this._numNodesV];
    _normals = new PVector[this._numNodesH][this._numNodesV];
    _springs = new ArrayList<DampedSpring>();

    dirH.normalize().mult( sepH);         // Valor de incremento en componente X
    dirV.normalize().mult( sepV);         // Valor de incremento en componente Y

    Particle p;
    PVector v = new PVector( 0, 0);
    
    PVector iniV = new PVector();
    iniV.set( iniPos);
    iniV.add( PVector.mult( dirH, -1*floor( _numNodesH/2)));
    iniV.add( PVector.mult( dirV, -1*floor( _numNodesV/2)));

    PVector pos = new PVector();
    pos.set( iniV);

    // Create Particles
    for( int i = 0; i < this._numNodesH; i++){
      // Volver al inicio de la superficie
      pos.set( iniV);

      for( int j = 0; j < this._numNodesV; j++){

        p = new Particle( pos, v, P_M, true, willBeClamped[i][j]);
        _nodes[i][j] = p;
        pos.add( dirV);
      }
      iniV.add( dirH);
    }

    DampedSpring spring;
    // p = p1
    Particle p2;
    boolean enabled;
    layoutType type = layoutType.BEND;

    // Create Springs for Bend Model
    // Recorrer nodos, hacia la derecha y abajo, crear Springs hacia la izquierda y arriba
    // Rojo
    if( layout == SpringLayout.BEND || layout == SpringLayout.STRUCTURAL_AND_BEND || layout == SpringLayout.SHEAR_AND_BEND || layout == SpringLayout.STRUCTURAL_AND_SHEAR_AND_BEND)
      enabled = true;
    else enabled = false;

    for( int i = 0; i < _numNodesH; i++){
      for( int j = 0; j < _numNodesV; j++){
        p = _nodes[i][j];
        if( (i-2) >= 0){
          p2 = _nodes[i-2][j];
          spring = new DampedSpring( p, p2, #ff0000, enabled, type);
          _springs.add( spring);
        }

        if( (j-2) >= 0){
          p2 = _nodes[i][j-2];
          spring = new DampedSpring( p, p2, #ff0000, enabled, type);
          _springs.add( spring);
        }
      }
    }

    // Create Springs for Shear Model
    // Verde

    type = layoutType.SHEAR;

    if( layout == SpringLayout.SHEAR || layout == SpringLayout.STRUCTURAL_AND_SHEAR || layout == SpringLayout.SHEAR_AND_BEND || layout == SpringLayout.STRUCTURAL_AND_SHEAR_AND_BEND)
      enabled = true;
    else enabled = false;

    for( int i = 1; i < _numNodesH; i++){
      for( int j = 0; j < _numNodesV; j++){
        p = _nodes[i][j];
        if( j-1 >= 0){
          p2 = _nodes[i-1][j-1];
          spring = new DampedSpring( p, p2, #00ff00, enabled, type);
          _springs.add( spring);
        }

        if( j+1 < _numNodesV){
          p2 = _nodes[i-1][j+1];
          spring = new DampedSpring( p, p2, #00ff00, enabled, type);
          _springs.add( spring);
        }
      }
    }

    // Create Springs for Struct Model
    // Azul

    type = layoutType.STRUCTURAL;

    if( layout == SpringLayout.STRUCTURAL || layout == SpringLayout.STRUCTURAL_AND_SHEAR || layout == SpringLayout.STRUCTURAL_AND_BEND || layout == SpringLayout.STRUCTURAL_AND_SHEAR_AND_BEND)
      enabled = true;
    else enabled = false;
    
    for( int i = 0; i < _numNodesH; i++){
      for( int j = 0; j < _numNodesV; j++){
        p = _nodes[i][j];
        if( i-1 >= 0){
          p2 = _nodes[i-1][j];
          spring = new DampedSpring( p, p2, #0000ff, enabled, type);
          _springs.add( spring);
        }
        if( j-1 >= 0){
          p2 = _nodes[i][j-1];
          spring = new DampedSpring( p, p2, #0000ff, enabled, type);
          _springs.add( spring);
        }
      }
    }
  }

  void updateSpringLayout( SpringLayout layout){

    // Primero Seteamos todos los Springs a false
    for( DampedSpring spring : this._springs)
      spring.setEnabled( false);


    ArrayList<layoutType> _types = new ArrayList<layoutType>();
    layoutType type = layoutType.STRUCTURAL;
    String s1 = ""+layout;
    String s2 = ""+type;

    // Check for STRUCTURAL
    if( match( s1, s2) != null){
      _types.add( type);
    }

    // Check for SHEAR
    type = layoutType.SHEAR;
    s2 = ""+type;
    if( match( s1, s2) != null){
      _types.add( type);
    }

    // Check for BEND
    type = layoutType.BEND;
    s2 = ""+type;
    if( match( s1, s2) != null){
      _types.add( type);
    }

    for( DampedSpring spring : _springs){
      for( layoutType ltype : _types){
        if( spring.getType() == ltype){
          spring.setEnabled( true);
        }
      }
    }
  }

  int getNumNodes()
  {
    return _numNodesH*_numNodesV;
  }

  int getNumBrokenSprings()
  {
    int broken = 0;
    for( DampedSpring s : this._springs){
      if( s.isBroken())
        broken++;
    }
    return broken;
  }

  int getNumSprings()
  {
    return _springs.size();
  }

  void updateNormals(){
    PVector dx = new PVector();
    PVector dy = new PVector();
    PVector normal;

    for( int i = 0; i < this._numNodesH; i++){
      for( int j = 0; j < this._numNodesV; j++){
        normal = new PVector();
        dx.set( PVector.sub( this._nodes[(i+1 <= _numNodesH-1) ? i+1 : i-1][j].getPos(), this._nodes[i][j].getPos()));
        dy.set( PVector.sub( this._nodes[i][(j+1 <= _numNodesV-1) ? j+1 : j-1].getPos(), this._nodes[i][j].getPos()));
        normal.set( dx.cross( dy).normalize());
        this._nodes[i][j].setNormal( normal);
      }
    }
  }

  void update(float simStep)
  {
    for( DampedSpring spring : _springs){
      spring.update();
    }

    // Check Broken
    for( DampedSpring spring : _springs){
      spring.checkBroken();
    }

    updateNormals();

    for( int i = 0; i < this._numNodesH; i++){
      for( int j = 0; j < this._numNodesV; j++){
        _nodes[i][j].update( simStep);
      }
    }
  }

  void computeCollisions(){
    for( int i = 0; i < this._numNodesH; i++){
      for( int j = 0; j < this._numNodesV; j++){
        computeCollisionsSpring( _nodes[i][j]);
      }
    }
  }

  void computeCollisionsSpring( Particle p){
    PVector dist = PVector.sub(p.getPos(), _ball.getPos());
    if( dist.mag() <= _ball.getRadius()){
      float lRep = _ball.getRadius();
      PVector distu = dist.normalize( null);

      float Fs = -_Ke*50 * ( dist.mag() - lRep);

      PVector Fs1 = PVector.mult( distu, Fs);
      PVector v = PVector.add( p.getVel(), Fs1.mult( Kd));
      _ball.addExternalForce( PVector.mult( Fs1, -1));
      p.addExternalForce( Fs1);
      // p.setVel( v);
    }
  }

  void render()
  {
    if (DRAW_MODE)
      renderWithSegments();
    else
      renderWithQuads();

    if( _displaySprings)
      renderSprings();
  }

  void renderSprings(){
    for( DampedSpring s : _springs)
      s.render();
  }

  void renderWithQuads()
  {
    int i, j;

    fill(this._color);
    stroke(#000000);
    strokeWeight(0.5);

    for (j = 0; j < _numNodesV - 1; j++)
    {
      beginShape(QUAD_STRIP);
      for (i = 0; i < _numNodesH; i++)
      {
        if ((_nodes[i][j] != null) && (_nodes[i][j+1] != null))
        {
          PVector pos1 = _nodes[i][j].getPos();
          PVector pos2 = _nodes[i][j+1].getPos();

          vertex(pos1.x, pos1.y, pos1.z);
          vertex(pos2.x, pos2.y, pos2.z);
        }
      }
      endShape();
    }
  }

  void renderWithSegments()
  {
    stroke(_color);
    strokeWeight(0.5);

    for (DampedSpring s : _springs) 
    {
      // if (s.isEnabled())
      if (!s.isBroken())
      {
        PVector pos1 = s.getParticle1().getPos();
        PVector pos2 = s.getParticle2().getPos();

        line(pos1.x, pos1.y, pos1.z, pos2.x, pos2.y, pos2.z);
      }
    }
  }
}
