// Damped spring between two particles:
//
// Fp1 = Fe - Fd
// Fp2 = -Fe + Fd = -(Fe - Fd) = -Fp1
//
//    Fe = Ke·(l - l0)·eN
//    Fd = -Kd·eN·v
//
//    e = s2 - s1  : current elongation vector between the particles
//    l = |e|      : current length
//    eN = e/l     : current normalized elongation vector
//    v = dl/dt    : rate of change of length

public class DampedSpring
{
  Particle _p1;     // First particle attached to the spring
  Particle _p2;     // Second particle attached to the spring

  float _l0;        // Rest length (m)
  float _l;         // Current length (m)
  float _v;         // Current rate of change of length (m/s)

  PVector _e;       // Current elongation vector (m)
  PVector _eN;      // Current normalized elongation vector (no units)
  PVector _F;       // Force applied by the spring on particle 1 (the force on particle 2 is -_F) (N)
  float _maxF;
  
  boolean _enabled; // To enable/disable the spring
  boolean _broken;  // To indicate when a Spring is broken
  color _c;

  layoutType _type;


  DampedSpring( Particle p1, Particle p2, color c, boolean enabled, layoutType type)
  {
    this._p1 = p1;
    this._p2 = p2;
    this._c = c;
    this._enabled = enabled;
    this._broken = false;

    this._l0 = PVector.sub(this._p1.getPos(), this._p2.getPos()).mag();

    this._F = new PVector( 0, 0, 0);
    this._maxF = MAX_FORCE;

    this._type = type;
  }

  Particle getParticle1()
  {
    return _p1;
  }

  Particle getParticle2()
  {
    return _p2;
  }

  layoutType getType(){
    return this._type;
  }

  void update()
  {
    if( this._enabled && !this._broken){
      float prevL = this._l;
      this._e = PVector.sub( _p1.getPos(), _p2.getPos());
      this._l = _e.mag();
      this._eN = _e.normalize( null);

      this._v = this._l - prevL;

      // Fs = -Ke*( l - l0)
      _F.add( PVector.mult( this._eN, -_Ke * (_l - this._l0)));

      // Fd = -Kd·eN·v
      _F.add( PVector.sub( this._p1.getVel(), this._p2.getVel()).mult( -_Kd));
      // _F.mult( Kd);

      applyForces();
    }
    // _F.set( 0, 0, 0); // Se actualiza en checkBroken()
  }

  void checkBroken(){
    if( this._enabled && this._F.mag() > this._maxF){
      this._broken = true;
      // _defOb.removeParticle( this._p1);
      // _defOb.removeParticle( this._p2);
    } else
      this._F.set( 0, 0, 0); // Se actualiza aqui para poder comprobar si se rompe o no, teniendo en cuenta el computo total de fuerzas sobre el muelle
  }

  void render(){
    if( this._enabled){
      color c = (!this._broken) ? this._c : #454545;
      stroke( c);
      strokeWeight( (this._type == layoutType.BEND) ? 10 : 5);
      line( this._p1.getPos().x, this._p1.getPos().y, this._p1.getPos().z, this._p2.getPos().x, this._p2.getPos().y, this._p2.getPos().z);
    }
  }

  void applyForces()
  {
    _p1.addExternalForce(_F.copy());
    _p2.addExternalForce(PVector.mult(_F, -1.0));
  }

  boolean isEnabled()
  {
    return _enabled;
  }

  boolean isBroken()
  {
    return this._broken;
  }

  void setEnabled( boolean enabled){
    this._enabled = enabled;
  }
}
